/*!

This library implements
[Unicode Canonical Combining Class](https://unicode.org/reports/tr44/#Canonical_Combining_Class_Values) detection.

```rust
use unicode_ccc::*;

assert_eq!(get_canonical_combining_class('A'), CanonicalCombiningClass::NotReordered);
assert_eq!(get_canonical_combining_class('\u{0A3C}'), CanonicalCombiningClass::Nukta);
assert_eq!(get_canonical_combining_class('\u{18A9}'), CanonicalCombiningClass::AboveLeft);
```

*/

#![no_std]

#![forbid(unsafe_code)]

/// The Unicode version.
pub const UNICODE_VERSION: (u8, u8, u8) = (16, 0, 0);

/// Character Canonical Combining Class.
#[derive(Clone, Copy, PartialEq, Debug)]
pub enum CanonicalCombiningClass {
    NotReordered = 0,
    Overlay = 1,
    HanReading = 6,
    Nukta = 7,
    KanaVoicing = 8,
    Virama = 9,
    // Hebrew
    CCC10 = 10,
    CCC11 = 11,
    CCC12 = 12,
    CCC13 = 13,
    CCC14 = 14,
    CCC15 = 15,
    CCC16 = 16,
    CCC17 = 17,
    CCC18 = 18,
    CCC19 = 19,
    CCC20 = 20,
    CCC21 = 21,
    CCC22 = 22,
    CCC23 = 23,
    CCC24 = 24,
    CCC25 = 25,
    CCC26 = 26,
    // Arabic
    CCC27 = 27,
    CCC28 = 28,
    CCC29 = 29,
    CCC30 = 30,
    CCC31 = 31,
    CCC32 = 32,
    CCC33 = 33,
    CCC34 = 34,
    CCC35 = 35,
    // Syriac
    CCC36 = 36,
    // Telugu
    CCC84 = 84,
    CCC91 = 91,
    // Thai
    CCC103 = 103,
    CCC107 = 107,
    // Lao
    CCC118 = 118,
    CCC122 = 122,
    // Tibetan
    CCC129 = 129,
    CCC130 = 130,
    CCC132 = 132,
    AttachedBelowLeft = 200,
    AttachedBelow = 202,
    AttachedAbove = 214,
    AttachedAboveRight = 216,
    BelowLeft = 218,
    Below = 220,
    BelowRight = 222,
    Left = 224,
    Right = 226,
    AboveLeft = 228,
    Above = 230,
    AboveRight = 232,
    DoubleBelow = 233,
    DoubleAbove = 234,
    IotaSubscript = 240,
}

/// Returns a Canonical Combining Class of a character.
///
/// Based on <https://www.unicode.org/Public/14.0.0/ucd/extracted/DerivedCombiningClass.txt>.
pub fn get_canonical_combining_class(c: char) -> CanonicalCombiningClass {
    use CanonicalCombiningClass::*;

    match c as u32 {
        0x0334..=0x0338 => Overlay,
        0x1CD4 => Overlay,
        0x1CE2..=0x1CE8 => Overlay,
        0x20D2..=0x20D3 => Overlay,
        0x20D8..=0x20DA => Overlay,
        0x20E5..=0x20E6 => Overlay,
        0x20EA..=0x20EB => Overlay,
        0x10A39 => Overlay,
        0x16AF0..=0x16AF4 => Overlay,
        0x1BC9E => Overlay,
        0x1D167..=0x1D169 => Overlay,
        0x16FF0..=0x16FF1 => HanReading,
        0x093C => Nukta,
        0x09BC => Nukta,
        0x0A3C => Nukta,
        0x0ABC => Nukta,
        0x0B3C => Nukta,
        0x0C3C => Nukta,
        0x0CBC => Nukta,
        0x1037 => Nukta,
        0x1B34 => Nukta,
        0x1BE6 => Nukta,
        0x1C37 => Nukta,
        0xA9B3 => Nukta,
        0x110BA => Nukta,
        0x11173 => Nukta,
        0x111CA => Nukta,
        0x11236 => Nukta,
        0x112E9 => Nukta,
        0x1133B..=0x1133C => Nukta,
        0x11446 => Nukta,
        0x114C3 => Nukta,
        0x115C0 => Nukta,
        0x116B7 => Nukta,
        0x1183A => Nukta,
        0x11943 => Nukta,
        0x11D42 => Nukta,
        0x1E94A => Nukta,
        0x3099..=0x309A => KanaVoicing,
        0x094D => Virama,
        0x09CD => Virama,
        0x0A4D => Virama,
        0x0ACD => Virama,
        0x0B4D => Virama,
        0x0BCD => Virama,
        0x0C4D => Virama,
        0x0CCD => Virama,
        0x0D3B..=0x0D3C => Virama,
        0x0D4D => Virama,
        0x0DCA => Virama,
        0x0E3A => Virama,
        0x0EBA => Virama,
        0x0F84 => Virama,
        0x1039..=0x103A => Virama,
        0x1714 => Virama,
        0x1715 => Virama,
        0x1734 => Virama,
        0x17D2 => Virama,
        0x1A60 => Virama,
        0x1B44 => Virama,
        0x1BAA => Virama,
        0x1BAB => Virama,
        0x1BF2..=0x1BF3 => Virama,
        0x2D7F => Virama,
        0xA806 => Virama,
        0xA82C => Virama,
        0xA8C4 => Virama,
        0xA953 => Virama,
        0xA9C0 => Virama,
        0xAAF6 => Virama,
        0xABED => Virama,
        0x10A3F => Virama,
        0x11046 => Virama,
        0x11070 => Virama,
        0x1107F => Virama,
        0x110B9 => Virama,
        0x11133..=0x11134 => Virama,
        0x111C0 => Virama,
        0x11235 => Virama,
        0x112EA => Virama,
        0x1134D => Virama,
        0x113CE => Virama,
        0x113CF => Virama,
        0x113D0 => Virama,
        0x11442 => Virama,
        0x114C2 => Virama,
        0x115BF => Virama,
        0x1163F => Virama,
        0x116B6 => Virama,
        0x1172B => Virama,
        0x11839 => Virama,
        0x1193D => Virama,
        0x1193E => Virama,
        0x119E0 => Virama,
        0x11A34 => Virama,
        0x11A47 => Virama,
        0x11A99 => Virama,
        0x11C3F => Virama,
        0x11D44..=0x11D45 => Virama,
        0x11D97 => Virama,
        0x11F41 => Virama,
        0x11F42 => Virama,
        0x1612F => Virama,
        0x05B0 => CCC10,
        0x05B1 => CCC11,
        0x05B2 => CCC12,
        0x05B3 => CCC13,
        0x05B4 => CCC14,
        0x05B5 => CCC15,
        0x05B6 => CCC16,
        0x05B7 => CCC17,
        0x05B8 => CCC18,
        0x05C7 => CCC18,
        0x05B9..=0x05BA => CCC19,
        0x05BB => CCC20,
        0x05BC => CCC21,
        0x05BD => CCC22,
        0x05BF => CCC23,
        0x05C1 => CCC24,
        0x05C2 => CCC25,
        0xFB1E => CCC26,
        0x064B => CCC27,
        0x08F0 => CCC27,
        0x064C => CCC28,
        0x08F1 => CCC28,
        0x064D => CCC29,
        0x08F2 => CCC29,
        0x0618 => CCC30,
        0x064E => CCC30,
        0x0619 => CCC31,
        0x064F => CCC31,
        0x061A => CCC32,
        0x0650 => CCC32,
        0x0651 => CCC33,
        0x0652 => CCC34,
        0x0670 => CCC35,
        0x0711 => CCC36,
        0x0C55 => CCC84,
        0x0C56 => CCC91,
        0x0E38..=0x0E39 => CCC103,
        0x0E48..=0x0E4B => CCC107,
        0x0EB8..=0x0EB9 => CCC118,
        0x0EC8..=0x0ECB => CCC122,
        0x0F71 => CCC129,
        0x0F72 => CCC130,
        0x0F7A..=0x0F7D => CCC130,
        0x0F80 => CCC130,
        0x0F74 => CCC132,
        0x0321..=0x0322 => AttachedBelow,
        0x0327..=0x0328 => AttachedBelow,
        0x1DD0 => AttachedBelow,
        0x1DCE => AttachedAbove,
        0x031B => AttachedAboveRight,
        0x0F39 => AttachedAboveRight,
        0x1D165..=0x1D166 => AttachedAboveRight,
        0x1D16E..=0x1D172 => AttachedAboveRight,
        0x1DFA => BelowLeft,
        0x302A => BelowLeft,
        0x0316..=0x0319 => Below,
        0x031C..=0x0320 => Below,
        0x0323..=0x0326 => Below,
        0x0329..=0x0333 => Below,
        0x0339..=0x033C => Below,
        0x0347..=0x0349 => Below,
        0x034D..=0x034E => Below,
        0x0353..=0x0356 => Below,
        0x0359..=0x035A => Below,
        0x0591 => Below,
        0x0596 => Below,
        0x059B => Below,
        0x05A2..=0x05A7 => Below,
        0x05AA => Below,
        0x05C5 => Below,
        0x0655..=0x0656 => Below,
        0x065C => Below,
        0x065F => Below,
        0x06E3 => Below,
        0x06EA => Below,
        0x06ED => Below,
        0x0731 => Below,
        0x0734 => Below,
        0x0737..=0x0739 => Below,
        0x073B..=0x073C => Below,
        0x073E => Below,
        0x0742 => Below,
        0x0744 => Below,
        0x0746 => Below,
        0x0748 => Below,
        0x07F2 => Below,
        0x07FD => Below,
        0x0859..=0x085B => Below,
        0x0899..=0x089B => Below,
        0x08CF..=0x08D3 => Below,
        0x08E3 => Below,
        0x08E6 => Below,
        0x08E9 => Below,
        0x08ED..=0x08EF => Below,
        0x08F6 => Below,
        0x08F9..=0x08FA => Below,
        0x0952 => Below,
        0x0F18..=0x0F19 => Below,
        0x0F35 => Below,
        0x0F37 => Below,
        0x0FC6 => Below,
        0x108D => Below,
        0x193B => Below,
        0x1A18 => Below,
        0x1A7F => Below,
        0x1AB5..=0x1ABA => Below,
        0x1ABD => Below,
        0x1ABF..=0x1AC0 => Below,
        0x1AC3..=0x1AC4 => Below,
        0x1ACA => Below,
        0x1B6C => Below,
        0x1CD5..=0x1CD9 => Below,
        0x1CDC..=0x1CDF => Below,
        0x1CED => Below,
        0x1DC2 => Below,
        0x1DCA => Below,
        0x1DCF => Below,
        0x1DF9 => Below,
        0x1DFD => Below,
        0x1DFF => Below,
        0x20E8 => Below,
        0x20EC..=0x20EF => Below,
        0xA92B..=0xA92D => Below,
        0xAAB4 => Below,
        0xFE27..=0xFE2D => Below,
        0x101FD => Below,
        0x102E0 => Below,
        0x10A0D => Below,
        0x10A3A => Below,
        0x10AE6 => Below,
        0x10EFD..=0x10EFF => Below,
        0x10F46..=0x10F47 => Below,
        0x10F4B => Below,
        0x10F4D..=0x10F50 => Below,
        0x10F83 => Below,
        0x10F85 => Below,
        0x1D17B..=0x1D182 => Below,
        0x1D18A..=0x1D18B => Below,
        0x1E4EE => Below,
        0x1E5EF => Below,
        0x1E8D0..=0x1E8D6 => Below,
        0x059A => BelowRight,
        0x05AD => BelowRight,
        0x1939 => BelowRight,
        0x302D => BelowRight,
        0x302E..=0x302F => Left,
        0x1D16D => Right,
        0x05AE => AboveLeft,
        0x18A9 => AboveLeft,
        0x1DF7..=0x1DF8 => AboveLeft,
        0x302B => AboveLeft,
        0x0300..=0x0314 => Above,
        0x033D..=0x0344 => Above,
        0x0346 => Above,
        0x034A..=0x034C => Above,
        0x0350..=0x0352 => Above,
        0x0357 => Above,
        0x035B => Above,
        0x0363..=0x036F => Above,
        0x0483..=0x0487 => Above,
        0x0592..=0x0595 => Above,
        0x0597..=0x0599 => Above,
        0x059C..=0x05A1 => Above,
        0x05A8..=0x05A9 => Above,
        0x05AB..=0x05AC => Above,
        0x05AF => Above,
        0x05C4 => Above,
        0x0610..=0x0617 => Above,
        0x0653..=0x0654 => Above,
        0x0657..=0x065B => Above,
        0x065D..=0x065E => Above,
        0x06D6..=0x06DC => Above,
        0x06DF..=0x06E2 => Above,
        0x06E4 => Above,
        0x06E7..=0x06E8 => Above,
        0x06EB..=0x06EC => Above,
        0x0730 => Above,
        0x0732..=0x0733 => Above,
        0x0735..=0x0736 => Above,
        0x073A => Above,
        0x073D => Above,
        0x073F..=0x0741 => Above,
        0x0743 => Above,
        0x0745 => Above,
        0x0747 => Above,
        0x0749..=0x074A => Above,
        0x07EB..=0x07F1 => Above,
        0x07F3 => Above,
        0x0816..=0x0819 => Above,
        0x081B..=0x0823 => Above,
        0x0825..=0x0827 => Above,
        0x0829..=0x082D => Above,
        0x0897..=0x0898 => Above,
        0x089C..=0x089F => Above,
        0x08CA..=0x08CE => Above,
        0x08D4..=0x08E1 => Above,
        0x08E4..=0x08E5 => Above,
        0x08E7..=0x08E8 => Above,
        0x08EA..=0x08EC => Above,
        0x08F3..=0x08F5 => Above,
        0x08F7..=0x08F8 => Above,
        0x08FB..=0x08FF => Above,
        0x0951 => Above,
        0x0953..=0x0954 => Above,
        0x09FE => Above,
        0x0F82..=0x0F83 => Above,
        0x0F86..=0x0F87 => Above,
        0x135D..=0x135F => Above,
        0x17DD => Above,
        0x193A => Above,
        0x1A17 => Above,
        0x1A75..=0x1A7C => Above,
        0x1AB0..=0x1AB4 => Above,
        0x1ABB..=0x1ABC => Above,
        0x1AC1..=0x1AC2 => Above,
        0x1AC5..=0x1AC9 => Above,
        0x1ACB..=0x1ACE => Above,
        0x1B6B => Above,
        0x1B6D..=0x1B73 => Above,
        0x1CD0..=0x1CD2 => Above,
        0x1CDA..=0x1CDB => Above,
        0x1CE0 => Above,
        0x1CF4 => Above,
        0x1CF8..=0x1CF9 => Above,
        0x1DC0..=0x1DC1 => Above,
        0x1DC3..=0x1DC9 => Above,
        0x1DCB..=0x1DCC => Above,
        0x1DD1..=0x1DF5 => Above,
        0x1DFB => Above,
        0x1DFE => Above,
        0x20D0..=0x20D1 => Above,
        0x20D4..=0x20D7 => Above,
        0x20DB..=0x20DC => Above,
        0x20E1 => Above,
        0x20E7 => Above,
        0x20E9 => Above,
        0x20F0 => Above,
        0x2CEF..=0x2CF1 => Above,
        0x2DE0..=0x2DFF => Above,
        0xA66F => Above,
        0xA674..=0xA67D => Above,
        0xA69E..=0xA69F => Above,
        0xA6F0..=0xA6F1 => Above,
        0xA8E0..=0xA8F1 => Above,
        0xAAB0 => Above,
        0xAAB2..=0xAAB3 => Above,
        0xAAB7..=0xAAB8 => Above,
        0xAABE..=0xAABF => Above,
        0xAAC1 => Above,
        0xFE20..=0xFE26 => Above,
        0xFE2E..=0xFE2F => Above,
        0x10376..=0x1037A => Above,
        0x10A0F => Above,
        0x10A38 => Above,
        0x10AE5 => Above,
        0x10D24..=0x10D27 => Above,
        0x10D69..=0x10D6D => Above,
        0x10EAB..=0x10EAC => Above,
        0x10F48..=0x10F4A => Above,
        0x10F4C => Above,
        0x10F82 => Above,
        0x10F84 => Above,
        0x11100..=0x11102 => Above,
        0x11366..=0x1136C => Above,
        0x11370..=0x11374 => Above,
        0x1145E => Above,
        0x16B30..=0x16B36 => Above,
        0x1D185..=0x1D189 => Above,
        0x1D1AA..=0x1D1AD => Above,
        0x1D242..=0x1D244 => Above,
        0x1E000..=0x1E006 => Above,
        0x1E008..=0x1E018 => Above,
        0x1E01B..=0x1E021 => Above,
        0x1E023..=0x1E024 => Above,
        0x1E026..=0x1E02A => Above,
        0x1E08F => Above,
        0x1E130..=0x1E136 => Above,
        0x1E2AE => Above,
        0x1E2EC..=0x1E2EF => Above,
        0x1E4EF => Above,
        0x1E5EE => Above,
        0x1E944..=0x1E949 => Above,
        0x0315 => AboveRight,
        0x031A => AboveRight,
        0x0358 => AboveRight,
        0x1DF6 => AboveRight,
        0x302C => AboveRight,
        0x1E4EC..=0x1E4ED => AboveRight,
        0x035C => DoubleBelow,
        0x035F => DoubleBelow,
        0x0362 => DoubleBelow,
        0x1DFC => DoubleBelow,
        0x035D..=0x035E => DoubleAbove,
        0x0360..=0x0361 => DoubleAbove,
        0x1DCD => DoubleAbove,
        0x0345 => IotaSubscript,
        _ => NotReordered,
    }
}
